#!/usr/bin/python3
# -- coding: utf-8 --

# csv2yalm4.py
# Autor: Roberto Plà Aragonés
# Inicio 26/12/2025
# Version 0.4 31/01/2022 19h:16
# Fecha: jue 01 ene 2026 19:16:43 CET

# Comentarios sobre el programa
# Convertir los datos de las plumas en notas markdown para una base de datos de obsidian

# Importación de módulos
import csv
import os
import unicodedata

# Configuración
archivo_entrada = 'estilograficas.csv'
carpeta_salida = 'salida_md'
mi_tiponota = 'fichapluma'
mi_plantilla = 'ficha_pluma'

if not os.path.exists(carpeta_salida):
    os.makedirs(carpeta_salida)

def normalizar_nombre_archivo(texto):
# Elimina tildes, eñes, caracteres especiales y espacios.
# Ejemplo: "Café más Añejo" -> "CafemasAnejo"

    # 1. Convertir a string y normalizar a formato NFD (separa letra de su tilde)
    texto = str(texto)
    texto = unicodedata.normalize('NFD', texto)
    
    # 2. Filtrar solo los caracteres que no sean marcas de combinación (tildes )
    # y que sean alfanuméricos
    texto_limpio = "".join(
        c for c in texto 
        if unicodedata.category(c) != 'Mn' and (c.isalnum() or c == ' ')
    )
    
    # 3. Eliminar espacios y devolver
    return texto_limpio.replace(" ", "")

try:
    with open(archivo_entrada, mode='r', encoding='utf-8') as csv_archivo:
        datos_csv = csv.DictReader(csv_archivo, delimiter=';', quotechar='"')
        
        for fila in datos_csv:
            columnas = list(fila.keys())
            
            # Generar el nombre del archivo ya saneado
            parte1 = normalizar_nombre_archivo(fila[columnas[0]])
            parte2 = normalizar_nombre_archivo(fila[columnas[1]])
            parte3 = normalizar_nombre_archivo(fila[columnas[2]])
            parte4 = normalizar_nombre_archivo(fila[columnas[3]])
            
            nombre_archivo = f"{parte1}_{parte2}.md"
            
            # Construir el contenido YAML
            contenido_yaml = f"---\ntiponota: {mi_tiponota}\nplantilla: {mi_plantilla}\n"
            for campo, valor in fila.items():
                contenido_yaml += f"{campo}: {valor}\n"
            contenido_yaml += "   - ficha_pluma\n   - estilografica\n   - coleccion\n"    
            contenido_yaml += "---\n"
            contenido_yaml += f"## {parte2} {parte3} {parte4}\n"
            contenido_yaml += f"\n\n### Entintado\n"
            contenido_yaml += f"\n\n### Estado\n"
                        
            # Guardar el archivo
            ruta_completa = os.path.join(carpeta_salida, nombre_archivo)
            with open(ruta_completa, mode='w', encoding='utf-8') as f_salida:
                f_salida.write(contenido_yaml)
                
    print(f"✅ Proceso completado. Archivos generados en: {carpeta_salida}")

except FileNotFoundError:
    print(f"❌ Error: No se encontró '{archivo_entrada}'")
except Exception as e:
    print(f"⚠️ Ocurrió un error: {e}")
